const mysqlConnection = require('../utils/database');

/**
 * Check for upcoming follow-up reminders for a specific user
 * Returns follow-ups due within the next 15 minutes
 */
const checkFollowUpReminders = async (req, res) => {
  try {
    const userEmail = req.headers.authorization?.replace('Bearer ', '');
    
    if (!userEmail) {
      return res.status(401).json({ success: false, message: 'Unauthorized' });
    }

    // Get follow-ups that are:
    // 1. Status 'N' (active)
    // 2. Assigned to this user
    // 3. Due within next 15 minutes
    // 4. Not yet notified
    const query = `
      SELECT 
        lc.id,
        lc.lead_id,
        lc.followupdate,
        lc.comments,
        lc.user,
        lcust.full_name AS customer_name
      FROM leads_comments lc
      LEFT JOIN leads_main lm ON lc.lead_id = lm.id
      LEFT JOIN leads_customers lcust ON lm.customer = lcust.id
      WHERE lc.status = 'N'
        AND lc.user = ?
        AND lc.followupdate IS NOT NULL
        AND lc.followupdate >= NOW()
        AND lc.followupdate <= DATE_ADD(NOW(), INTERVAL 15 MINUTE)
        AND (lc.notified IS NULL OR lc.notified = 0)
      ORDER BY lc.followupdate ASC
    `;

    mysqlConnection.query(query, [userEmail], (err, results) => {
      if (err) {
        console.error('[FollowUpReminders] Database error:', err);
        return res.status(500).json({ success: false, message: 'Database error' });
      }

      return res.json({
        success: true,
        reminders: results || [],
        count: results ? results.length : 0,
      });
    });

  } catch (error) {
    console.error('[FollowUpReminders] Error:', error);
    return res.status(500).json({ success: false, message: 'Server error' });
  }
};

module.exports = {
  checkFollowUpReminders,
};
