const mysqlConnection = require('../../utils/database');
const fs = require('fs');
const path = require('path');

// Table schema suggestion (MySQL):
// CREATE TABLE call_logs (
//   id INT AUTO_INCREMENT PRIMARY KEY,
//   lead_id VARCHAR(64) NOT NULL,
//   user_email VARCHAR(255) NOT NULL,
//   phone VARCHAR(50),
//   start_time DATETIME NOT NULL,
//   end_time DATETIME NULL,
//   duration_seconds INT NULL,
//   recording_url VARCHAR(255) NULL,
//   notes TEXT NULL,
//   created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
//   updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
// );

// POST /call-log
exports.createCallLog = (req, res) => {
  const { lead_id, user_email, phone, start_time } = req.body;
  if (!lead_id || !user_email || !start_time) {
    return res
      .status(400)
      .json({ success: false, message: 'lead_id, user_email, start_time required' });
  }

  const sql = 'INSERT INTO call_logs (lead_id, user_email, phone, start_time) VALUES (?,?,?,?)';
  mysqlConnection.query(sql, [lead_id, user_email, phone || null, start_time], (err, result) => {
    if (err) {
      console.error('createCallLog error', err);
      return res.status(500).json({ success: false, message: 'DB error' });
    }
    return res.json({ success: true, id: result.insertId });
  });
};

// PUT /call-log/:id
// Only handles end_time, duration_seconds and optional notes.
// recording_url is handled separately by addCallRecording.
exports.updateCallLog = (req, res) => {
  const { id } = req.params;
  const { end_time, duration_seconds, notes } = req.body;

  if (!id) {
    return res.status(400).json({ success: false, message: 'id param required' });
  }
  if (!end_time || typeof duration_seconds === 'undefined') {
    return res
      .status(400)
      .json({ success: false, message: 'end_time and duration_seconds required' });
  }

  const fields = ['end_time = ?', 'duration_seconds = ?'];
  const params = [end_time, duration_seconds];

  // notes is optional – only update if present
  if (typeof notes !== 'undefined') {
    fields.push('notes = ?');
    params.push(notes || null);
  }

  const sql = `UPDATE call_logs SET ${fields.join(', ')} WHERE id = ?`;
  params.push(id);

  mysqlConnection.query(sql, params, (err, result) => {
    if (err) {
      console.error('updateCallLog error', err);
      return res.status(500).json({ success: false, message: 'DB error' });
    }
    if (result.affectedRows === 0) {
      return res.status(404).json({ success: false, message: 'Call log not found' });
    }
    return res.json({ success: true, message: 'Call log updated' });
  });
};

// PUT /call-log/:id/recording
// Accept base64 audio, write file, and update recording_url in call_logs
exports.addCallRecording = (req, res) => {
  const { id } = req.params;
  const { base64, filename } = req.body;

  if (!id) {
    return res.status(400).json({ success: false, message: 'id param required' });
  }
  if (!base64) {
    return res.status(400).json({ success: false, message: 'base64 audio required' });
  }

  const safeName =
    filename && /^[a-zA-Z0-9_.-]+$/.test(filename)
      ? filename
      : `call_${id}_${Date.now()}.m4a`;

  const uploadDir = path.join(__dirname, '../../uploads/call_recordings');

  try {
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }

    const filePath = path.join(uploadDir, safeName);
    const buffer = Buffer.from(base64, 'base64');
    fs.writeFileSync(filePath, buffer);

    const relativeUrl = `/uploads/call_recordings/${safeName}`;

    const sql = 'UPDATE call_logs SET recording_url = ? WHERE id = ?';
    mysqlConnection.query(sql, [relativeUrl, id], (err, result) => {
      if (err) {
        console.error('addCallRecording db error', err);
        return res.status(500).json({ success: false, message: 'DB error' });
      }
      if (result.affectedRows === 0) {
        return res.status(404).json({ success: false, message: 'Call log not found' });
      }
      return res.json({
        success: true,
        message: 'Recording stored',
        recording_url: relativeUrl,
      });
    });
  } catch (e) {
    console.error('addCallRecording error', e);
    return res.status(500).json({ success: false, message: 'Failed to store recording' });
  }
};
