require('dotenv').config();

const http = require('http');
const express = require('express');
const bodyParser = require('body-parser');
const cors = require('cors');
const qbRoutes = require('./routes/qb');
const { Server } = require('socket.io');
const { setIO } = require('./services/NotificationService');

const app = express();
const server = http.createServer(app);

// Global logger for debugging incoming requests (MUST BE AT THE TOP)
app.use((req, res, next) => {
  console.log(`\n--- [${new Date().toISOString()}] Incoming Request ---`);
  console.log(`${req.method} ${req.url}`);
  next();
});

const io = new Server(server, {
  cors: {
    origin: [
      'https://www.prosale.sale',
      'http://localhost:3000',
      'http://localhost:8081',
      'http://192.168.1.131:8081',
      'https://posttabetic-unreciprocal-abby.ngrok-free.dev',
      'http://192.168.1.131:8081',
      /^exp:\/\/.*/, // Allow Expo development URLs
    ],
    methods: ['GET', 'PUT', 'POST', 'DELETE'],
    allowedHeaders: ['Content-Type', 'Authorization'],
    credentials: true
  },
  transports: ['websocket', 'polling']
});

// Initialize Socket.IO in NotificationService
setIO(io);

app.set('io', io);

// CORS configuration
const corsOptions = {
  origin: [
    'https://www.prosale.sale',
    'http://localhost:3000',
    'http://localhost:8081',
    'http://192.168.1.131:8081',
  ],
  methods: ['GET', 'PUT', 'POST', 'DELETE'],
  allowedHeaders: ['Content-Type', 'Authorization'],
  credentials: true
};

app.use(cors(corsOptions));
app.use(express.json({ limit: '20mb' }));

// Meta Webhook Routes (must be before other routes)
const { verifyWebhook, receiveWebhook } = require('./Controller/LeadM/MetaWebhook');
app.get('/webhook', verifyWebhook);
app.post('/webhook', receiveWebhook);

const port = 4000;

app.use(qbRoutes);

app.get('/', (req, res) => {
  res.send('Hello, Backend is running smoothly!');
});


io.on('connection', (socket) => {
  console.log(`✅ Socket.IO client connected: ${socket.id}`);

  socket.on('register', (userId) => {
    const roomName = `user_${userId}`;
    socket.join(roomName);
    console.log(`📍 User ${userId} registered for notifications in room: ${roomName}`);
    console.log(`   Total rooms: ${Array.from(io.sockets.sockets.get(socket.id)?._rooms || []).length}`);
    socket.emit('registered', { userId, roomName });
  });

  socket.on('disconnect', () => {
    console.log(`❌ Socket.IO client disconnected: ${socket.id}`);
  });

  socket.on('error', (error) => {
    console.error('⚠️ Socket.IO error:', error);
  });
});

// Debug: Log all active socket connections
setInterval(() => {
  const socketCount = io.sockets.sockets.size;
  if (socketCount > 0) {
    const rooms = Array.from(io.sockets.adapter.rooms.keys()).filter(room => !room.includes('/'));
    console.log(`[DEBUG] Active sockets: ${socketCount}, Rooms: ${rooms.join(', ')}`);
  }
}, 30000);

// Start server
server.listen(port, '0.0.0.0', () => {
  console.log(`✅ Server is running on port ${port}`);
  console.log(`✅ Socket.IO is ready on the same port (${port})`);
  console.log('Environment:', process.env.NODE_ENV || 'development');

  // Start scheduler AFTER server is listening
  const { startFollowUpScheduler } = require('./Controller/Dashboard/FollowUpReminder');
  startFollowUpScheduler(io);
});

// Handle server errors
server.on('error', (err) => {
  console.error('Server error:', err.message);
});

// Handle process errors
process.on('uncaughtException', (err) => {
  console.error('Uncaught exception:', err);
  process.exit(1);
});

process.on('unhandledRejection', (reason, promise) => {
  console.error('Unhandled rejection at:', promise, 'reason:', reason);
});

module.exports = { app, server, io };
