// Setup notification database tables
const mysql = require('mysql2/promise');

async function setupNotificationTables() {
  try {
    const connection = await mysql.createConnection({
      host: 'localhost',
      user: 'root',
      password: '',
      database: 'prosale',
    });

    console.log('Connected to database...');

    // Create expo_push_tokens table
    const expoTableSQL = `
      CREATE TABLE IF NOT EXISTS expo_push_tokens (
        id INT PRIMARY KEY AUTO_INCREMENT,
        user_id INT NOT NULL UNIQUE,
        token VARCHAR(500) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        INDEX idx_user_id (user_id)
      )
    `;

    await connection.query(expoTableSQL);
    console.log('✅ expo_push_tokens table created/verified');

    // Verify lead_subscription table structure
    const webPushTableSQL = `
      CREATE TABLE IF NOT EXISTS lead_subscription (
        id INT PRIMARY KEY AUTO_INCREMENT,
        username VARCHAR(191) NOT NULL UNIQUE,
        subscription LONGTEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_username (username)
      )
    `;

    await connection.query(webPushTableSQL);
    console.log('✅ lead_subscription table created/verified');

    // Create web_notifications table for tracking web push notifications
    const webNotificationsSQL = `
      CREATE TABLE IF NOT EXISTS web_notifications (
        id INT PRIMARY KEY AUTO_INCREMENT,
        username VARCHAR(191) NOT NULL,
        title VARCHAR(255) NOT NULL,
        body TEXT NOT NULL,
        lead_id INT,
        type ENUM('lead', 'followup', 'reminder') DEFAULT 'lead',
        status ENUM('pending', 'sent', 'failed') DEFAULT 'pending',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        sent_at TIMESTAMP NULL,
        FOREIGN KEY (username) REFERENCES users(name),
        INDEX idx_status (status),
        INDEX idx_created_at (created_at)
      )
    `;

    await connection.query(webNotificationsSQL);
    console.log('✅ web_notifications table created/verified');

    // Create mobile_notifications table for tracking mobile notifications
    const mobileNotificationsSQL = `
      CREATE TABLE IF NOT EXISTS mobile_notifications (
        id INT PRIMARY KEY AUTO_INCREMENT,
        user_id INT NOT NULL,
        title VARCHAR(255) NOT NULL,
        body TEXT NOT NULL,
        lead_id INT,
        type ENUM('lead', 'followup', 'reminder') DEFAULT 'lead',
        status ENUM('pending', 'sent', 'failed') DEFAULT 'pending',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        sent_at TIMESTAMP NULL,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        INDEX idx_status (status),
        INDEX idx_created_at (created_at)
      )
    `;

    await connection.query(mobileNotificationsSQL);
    console.log('✅ mobile_notifications table created/verified');

    // Create a real_time_notifications table for Socket.IO tracking
    const realtimeNotificationsSQL = `
      CREATE TABLE IF NOT EXISTS real_time_notifications (
        id INT PRIMARY KEY AUTO_INCREMENT,
        user_id INT,
        username VARCHAR(191),
        title VARCHAR(255) NOT NULL,
        body TEXT NOT NULL,
        lead_id INT,
        notification_type ENUM('web', 'mobile', 'email') NOT NULL,
        delivered_at TIMESTAMP NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_id (user_id),
        INDEX idx_created_at (created_at)
      )
    `;

    await connection.query(realtimeNotificationsSQL);
    console.log('✅ real_time_notifications table created/verified');

    console.log('\n✅ All notification tables are ready!');

    await connection.end();
    return true;
  } catch (error) {
    console.error('❌ Error setting up notification tables:', error.message);
    return false;
  }
}

// Run setup
setupNotificationTables().then((success) => {
  process.exit(success ? 0 : 1);
});
